--------
-- Simple camera script, using velocity to rotate camera.
--------

local cfg = ac.INIConfig.scriptSettings():mapSection('SETTINGS', {
  GFORCES = 0
})

-- Options for two different cameras (all distances are in meters):
local maximumCameraAngle = { 54, 68 }  -- degress

local function smoothed(value, smoothing)
  local lag = 1 - 1 / smoothing
  return function (target, dt)
    if target then
      value = math.applyLag(value, target, lag, dt)
    end
    return value
  end
end

-- Smoothed camera rotation:
local carVelocity = smoothed(vec3(), 40)

-- Smoothed looking-around direction:
local lookDirection = smoothed(0, 10)

-- Some forces shifting camera a bit:
local gForceX = smoothed(0, 50)
local gForceZ = smoothed(0, 50)

-- Will be called each frame:
-- Note: `dt` is time passed since last frame, `cameraIndex` is 1 or 2, depending on which camera is
-- chosen.
function script.update(dt, cameraIndex)
  -- Get AC camera parameters with some corrections to be somewhat compatible:
  local cameraParameters = ac.getCameraParameters(cameraIndex)
  local distance = cameraParameters.distance + 1.6
  local height = cameraParameters.height - 0.3
  local pitchAngle = cameraParameters.pitch - 5

  local carPos = car.transform.position -- using CoG for pivot
  local speedMult = math.saturateN(car.speedMs - 1) -- turning everying off for stationary cars

  -- Smoothing out velocity
  local velocitySmooth = carVelocity(car.velocity, dt)
  local speedSmooth = #velocitySmooth

  -- Get rotation coefficient, from -1 to 1, based on X-component of local velocity (that’s what dot is for)
  -- and taking absolute speed into account as well:
  local velocityX = math.clamp(
    -math.dot(car.side, velocitySmooth) / math.max(speedSmooth, 0.1) * math.pow(speedSmooth, 0.5) / 10, -1, 1) * speedMult

  -- Camera angle for given coefficient:
  local cameraAngle = velocityX * math.radians(maximumCameraAngle[cameraIndex])

  -- Extra thing for joystick support:
  local joystickLook = ac.getJoystickLook()
  cameraAngle = cameraAngle + lookDirection(
    (ac.looksLeft() and ac.looksRight() or ac.looksBehind()) and math.sign(lookDirection()) or
    ac.looksLeft() and 0.5 or
    ac.looksRight() and -0.5 or
    joystickLook ~= nil and joystickLook.x or 0, dt) * math.pi

  -- Sine and cosine for camera angle
  local sin, cos = math.sin(cameraAngle), math.cos(cameraAngle)

  -- Up direction for camera (could be used for horizon lock):
  local cameraUp = (car.up + vec3(0, 3, 0)):normalize()

  -- Set camera position:
  ac.Camera.position = carPos
    + car.side * (gForceX(car.acceleration.x * speedMult, dt) * -0.15 * cfg.GFORCES)
    + (car.side * sin - car.look * cos) * (distance * (1 + gForceZ(car.acceleration.z * speedMult, dt) * 0.05 * cfg.GFORCES))
    + vec3(0, height, 0)

  -- Find camera look
  local cameraLookPosOffset = car.look + car.up * (1 - math.abs(lookDirection()))
  local cameraLook = (carPos + cameraLookPosOffset - ac.Camera.position):normalize()

  -- -- Use for `pitchAngle`:
  cameraLook:rotate(quat.fromAngleAxis(math.rad(-pitchAngle), car.side))

  -- Set camera look:
  ac.Camera.direction = cameraLook

  -- Set other parameters:
  ac.Camera.up = cameraUp
  ac.Camera.fov = 60
end